<?php
define('AYENEH_ADMIN', true);
require_once 'php/autoload.php';

$pageTitle = 'تنظیمات سیستم';
$db = Database::getInstance();
$session = Session::getInstance();
$auth = Auth::getInstance();

// بررسی دسترسی Super Admin
if (!$auth->hasRole('super_admin')) {
    $session->setError('شما به این بخش دسترسی ندارید.');
    redirect('index.php');
}

$errors = [];
$success = false;

// دریافت تنظیمات فعلی
$settings = [];
$settingsData = $db->fetchAll("SELECT * FROM system_settings ORDER BY setting_key");
foreach ($settingsData as $setting) {
    $settings[$setting['setting_key']] = $setting['setting_value'];
}

// مقادیر پیش‌فرض
$defaultSettings = [
    'site_name' => 'آینه - Ayeneh',
    'site_url' => 'https://ayeneh.app',
    'admin_email' => 'admin@ayeneh.app',
    'support_email' => 'support@ayeneh.app',
    'items_per_page' => '20',
    'max_login_attempts' => '5',
    'login_lockout_minutes' => '15',
    'session_lifetime' => '3600',
    'enable_registration' => '1',
    'enable_notifications' => '1',
    'smtp_host' => '',
    'smtp_port' => '587',
    'smtp_username' => '',
    'smtp_password' => '',
    'smtp_encryption' => 'tls',
    'ai_provider' => 'openai',
    'ai_api_key' => '',
    'ai_model' => 'gpt-4',
    'debug_mode' => '0',
    'maintenance_mode' => '0'
];

// ترکیب تنظیمات پیش‌فرض با موجود
$settings = array_merge($defaultSettings, $settings);

if (isPost()) {
    // دریافت تنظیمات جدید
    $newSettings = [
        'site_name' => sanitize(post('site_name')),
        'site_url' => sanitize(post('site_url')),
        'admin_email' => sanitize(post('admin_email')),
        'support_email' => sanitize(post('support_email')),
        'items_per_page' => (int)post('items_per_page'),
        'max_login_attempts' => (int)post('max_login_attempts'),
        'login_lockout_minutes' => (int)post('login_lockout_minutes'),
        'session_lifetime' => (int)post('session_lifetime'),
        'enable_registration' => post('enable_registration', 0),
        'enable_notifications' => post('enable_notifications', 0),
        'smtp_host' => sanitize(post('smtp_host')),
        'smtp_port' => (int)post('smtp_port'),
        'smtp_username' => sanitize(post('smtp_username')),
        'smtp_encryption' => post('smtp_encryption'),
        'ai_provider' => post('ai_provider'),
        'ai_api_key' => sanitize(post('ai_api_key')),
        'ai_model' => sanitize(post('ai_model')),
        'debug_mode' => post('debug_mode', 0),
        'maintenance_mode' => post('maintenance_mode', 0)
    ];

    // بروزرسانی رمز SMTP فقط اگر تغییر کرده باشد
    if (post('smtp_password')) {
        $newSettings['smtp_password'] = sanitize(post('smtp_password'));
    }

    // اعتبارسنجی
    if (!isValidEmail($newSettings['admin_email'])) {
        $errors[] = 'فرمت ایمیل مدیر نامعتبر است.';
    }

    if (!isValidEmail($newSettings['support_email'])) {
        $errors[] = 'فرمت ایمیل پشتیبانی نامعتبر است.';
    }

    if (empty($errors)) {
        try {
            $db->beginTransaction();

            foreach ($newSettings as $key => $value) {
                // بررسی وجود تنظیم
                $existing = $db->fetchOne("SELECT id FROM system_settings WHERE setting_key = ?", [$key]);

                if ($existing) {
                    // بروزرسانی
                    $db->update('system_settings', [
                        'setting_value' => $value,
                        'updated_by' => $auth->getUserId(),
                        'updated_at' => date('Y-m-d H:i:s')
                    ], 'setting_key = ?', [$key]);
                } else {
                    // ایجاد
                    $db->insert('system_settings', [
                        'setting_key' => $key,
                        'setting_value' => $value,
                        'setting_type' => is_numeric($value) ? 'number' : 'string',
                        'updated_by' => $auth->getUserId()
                    ]);
                }
            }

            $db->commit();

            $success = true;
            $settings = $newSettings;
            $session->setSuccess('تنظیمات با موفقیت بروزرسانی شد.');

        } catch (Exception $e) {
            $db->rollback();
            $errors[] = 'خطا در بروزرسانی تنظیمات: ' . $e->getMessage();
            logError('Error updating settings: ' . $e->getMessage());
        }
    }
}

include 'php/includes/header.php';
?>


<div class="row clearfix">
    <div class="col-lg-12">
        <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                <i class="material-icons">check_circle</i>
                تنظیمات با موفقیت بروزرسانی شد.
            </div>
        <?php endif; ?>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <strong>خطاها:</strong>
                <ul class="m-b-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo h($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">

            <!-- تنظیمات عمومی -->
            <div class="card">
                <div class="header bg-cyan">
                    <h2 class="text-white">تنظیمات عمومی</h2>
                </div>
                <div class="body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>نام سایت</label>
                                <div class="form-line">
                                    <input type="text" name="site_name" class="form-control"
                                           value="<?php echo h($settings['site_name']); ?>" required>
                                </div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label>آدرس سایت</label>
                                <div class="form-line">
                                    <input type="url" name="site_url" class="form-control"
                                           value="<?php echo h($settings['site_url']); ?>" required>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>ایمیل مدیر</label>
                                <div class="form-line">
                                    <input type="email" name="admin_email" class="form-control"
                                           value="<?php echo h($settings['admin_email']); ?>" required>
                                </div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label>ایمیل پشتیبانی</label>
                                <div class="form-line">
                                    <input type="email" name="support_email" class="form-control"
                                           value="<?php echo h($settings['support_email']); ?>" required>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label>تعداد آیتم در هر صفحه</label>
                                <div class="form-line">
                                    <input type="number" name="items_per_page" class="form-control"
                                           value="<?php echo h($settings['items_per_page']); ?>" min="10" max="100">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-4">
                            <div class="form-group">
                                <label>مدت زمان Session (ثانیه)</label>
                                <div class="form-line">
                                    <input type="number" name="session_lifetime" class="form-control"
                                           value="<?php echo h($settings['session_lifetime']); ?>" min="600" max="86400">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-4">
                            <div class="form-group">
                                <label>&nbsp;</label>
                                <div>
                                    <input type="checkbox" name="enable_registration" id="enable_registration"
                                           value="1" <?php echo $settings['enable_registration'] ? 'checked' : ''; ?>>
                                    <label for="enable_registration">فعال بودن ثبت‌نام</label>
                                </div>
                                <div>
                                    <input type="checkbox" name="enable_notifications" id="enable_notifications"
                                           value="1" <?php echo $settings['enable_notifications'] ? 'checked' : ''; ?>>
                                    <label for="enable_notifications">فعال بودن نوتیفیکیشن</label>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- تنظیمات امنیتی -->
            <div class="card">
                <div class="header bg-orange">
                    <h2 class="text-white">تنظیمات امنیتی</h2>
                </div>
                <div class="body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>حداکثر تلاش ناموفق برای ورود</label>
                                <div class="form-line">
                                    <input type="number" name="max_login_attempts" class="form-control"
                                           value="<?php echo h($settings['max_login_attempts']); ?>" min="3" max="10">
                                </div>
                                <small class="text-muted">بعد از این تعداد، کاربر برای مدتی قفل می‌شود</small>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label>مدت قفل شدن (دقیقه)</label>
                                <div class="form-line">
                                    <input type="number" name="login_lockout_minutes" class="form-control"
                                           value="<?php echo h($settings['login_lockout_minutes']); ?>" min="5" max="60">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- تنظیمات ایمیل (SMTP) -->
            <div class="card">
                <div class="header bg-blue">
                    <h2 class="text-white">تنظیمات SMTP (ارسال ایمیل)</h2>
                </div>
                <div class="body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>SMTP Host</label>
                                <div class="form-line">
                                    <input type="text" name="smtp_host" class="form-control"
                                           value="<?php echo h($settings['smtp_host']); ?>"
                                           placeholder="smtp.gmail.com">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-3">
                            <div class="form-group">
                                <label>SMTP Port</label>
                                <div class="form-line">
                                    <input type="number" name="smtp_port" class="form-control"
                                           value="<?php echo h($settings['smtp_port']); ?>">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-3">
                            <div class="form-group">
                                <label>رمزنگاری</label>
                                <select name="smtp_encryption" class="form-control">
                                    <option value="tls" <?php echo $settings['smtp_encryption'] === 'tls' ? 'selected' : ''; ?>>TLS</option>
                                    <option value="ssl" <?php echo $settings['smtp_encryption'] === 'ssl' ? 'selected' : ''; ?>>SSL</option>
                                    <option value="none" <?php echo $settings['smtp_encryption'] === 'none' ? 'selected' : ''; ?>>بدون رمزنگاری</option>
                                </select>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>SMTP Username</label>
                                <div class="form-line">
                                    <input type="text" name="smtp_username" class="form-control"
                                           value="<?php echo h($settings['smtp_username']); ?>">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <label>SMTP Password</label>
                                <div class="form-line">
                                    <input type="password" name="smtp_password" class="form-control"
                                           placeholder="برای عدم تغییر خالی بگذارید">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- تنظیمات هوش مصنوعی -->
            <div class="card">
                <div class="header bg-purple">
                    <h2 class="text-white">تنظیمات هوش مصنوعی</h2>
                </div>
                <div class="body">
                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label>ارائه‌دهنده AI</label>
                                <select name="ai_provider" class="form-control">
                                    <option value="openai" <?php echo $settings['ai_provider'] === 'openai' ? 'selected' : ''; ?>>OpenAI</option>
                                    <option value="anthropic" <?php echo $settings['ai_provider'] === 'anthropic' ? 'selected' : ''; ?>>Anthropic (Claude)</option>
                                    <option value="google" <?php echo $settings['ai_provider'] === 'google' ? 'selected' : ''; ?>>Google (Gemini)</option>
                                </select>
                            </div>
                        </div>

                        <div class="col-md-4">
                            <div class="form-group">
                                <label>مدل AI</label>
                                <div class="form-line">
                                    <input type="text" name="ai_model" class="form-control"
                                           value="<?php echo h($settings['ai_model']); ?>"
                                           placeholder="gpt-4, claude-3-opus, gemini-pro">
                                </div>
                            </div>
                        </div>

                        <div class="col-md-4">
                            <div class="form-group">
                                <label>API Key</label>
                                <div class="form-line">
                                    <input type="password" name="ai_api_key" class="form-control"
                                           value="<?php echo h($settings['ai_api_key']); ?>"
                                           placeholder="sk-...">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- حالت‌های سیستم -->
            <div class="card">
                <div class="header bg-red">
                    <h2 class="text-white">حالت‌های سیستم</h2>
                </div>
                <div class="body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <input type="checkbox" name="debug_mode" id="debug_mode"
                                       value="1" <?php echo $settings['debug_mode'] ? 'checked' : ''; ?>>
                                <label for="debug_mode">حالت Debug (فقط برای توسعه)</label>
                                <p class="text-muted">در production باید غیرفعال باشد</p>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="form-group">
                                <input type="checkbox" name="maintenance_mode" id="maintenance_mode"
                                       value="1" <?php echo $settings['maintenance_mode'] ? 'checked' : ''; ?>>
                                <label for="maintenance_mode">حالت تعمیر و نگهداری</label>
                                <p class="text-muted">در این حالت، فقط ادمین‌ها می‌توانند وارد شوند</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="row">
                <div class="col-md-12">
                    <button type="submit" class="btn-action btn-save waves-effect">
                        <i class="material-icons">save</i>
                        <span>ذخیره تنظیمات</span>
                    </button>

                    <a href="index.php" class="btn-action waves-effect">
                        <i class="material-icons">cancel</i>
                        <span>انصراف</span>
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<?php include 'php/includes/footer.php'; ?>
